/** @file   SlimeAlien.cpp
 * @brief   Implementation of SlimeAlien class.
 * @version $Revision: 1.1 $
 * @date    $Date: 2006/05/08 21:34:04 $
 * @author  Tomi Lamminsaari
 */

#include "SlimeAlien.h"
#include "GfxManager.h"
#include "GameAnims.h"
#include "WarGlobals.h"
#include "www_map.h"
#include "GfxId.h"
#include "AnimId.h"
#include "RedrawQueue.h"
#include "SoundSamples.h"
#include "SmallWormAI.h"
#include "Settings.h"
using namespace eng2d;

namespace WeWantWar {

///
/// Constants, datatypes and static methods
/// ============================================================================




///
/// Constructors, destructor and operators
/// ============================================================================

/** Default constructor.
 */
SlimeAlien::SlimeAlien() :
  Alien()
{
  const Animation& anim = GameAnims::findAnimation( AnimId::KSlimeAlien, GameAnims::EWalk );
  m_animation = anim;
  
  int oid = ObjectID::TYPE_SLIMEALIEN;
  this->boundingSphere( Settings::floatObjProp(oid, "bounding_sphere:") );
  this->setArmor( Settings::floatObjProp(oid, "armor:") );
  
  this->setCollisionPoint( 0, Vec2D( 0,-10) );
  this->setCollisionPoint( 1, Vec2D( 0, 10) );
  this->setCollisionPoint( 2, Vec2D( -10, 0) );
  this->setCollisionPoint( 3, Vec2D( 10, 0) );
  this->setCounter( Alien::RELOAD_COUNTER_INDEX, 1 );
  
  this->setController( new SmallWormAI(this) );
}



/** Destructor.
 */
SlimeAlien::~SlimeAlien()
{
}



///
/// Methods inhertited from the base class(es)
/// ============================================================================

void SlimeAlien::update()
{
  if ( this->state() == GameObject::STATE_KILLED ) {
    return;
  }
  
  if ( this->state() == GameObject::STATE_DYING ) {
    if ( m_animation.paused() == true ) {
      this->state( GameObject::STATE_KILLED );
      this->setCorrectAnimation( GameAnims::EKilled );
    }
    return;
  }
//  this->setCounter( RELOAD_COUNTER_INDEX, 20 );
  
  GameObject* player = WarGlobals::pObjTable->pPlayer;
  Vec2D distVec = player->position() - this->position();
  if ( distVec.length() < this->boundingSphere() ) {
    this->setCorrectAnimation( GameAnims::EShoot );
    this->attack();
    
  } else {
    this->setCorrectAnimation( GameAnims::EWalk );
    BaseController* controller = this->getController();
    controller->update();
    this->changeAngle( controller->turn() );
    
    Vec2D mvec( 0, -Settings::floatObjProp(ObjectID::TYPE_SLIMEALIEN, "spd_walk:") );
    mvec.rotate( this->angle() );
    this->move( mvec );
  }
}

void SlimeAlien::redraw( RedrawQueue* aQueue )
{
  if ( this->hidden() == false ) {
    BITMAP* sprite = m_animation.currentFrame().asBitmap();
    Vec2D spPos = this->position() - Vec2D( Map::scrollX, Map::scrollY );
    spPos -= Vec2D( sprite->w/2, sprite->h/2 );
    aQueue->add( RedrawQueue::PRI_NORMAL, spPos.intX(), spPos.intY(),
                 RedrawQueue::SPTYPE_SPRITE, RedrawQueue::BMODE_TRANS,
                 50, sprite );
  }
}

bool SlimeAlien::hitByBullet( Bullet* aBullet )
{
  if ( aBullet->iType != Bullet::EFlameThrower ) {
    Vec2D dirV = aBullet->velocity();
    dirV *= 0.5;
    ParticleBlood* blood = new ParticleBlood( this->position(),
        dirV, 15, Color(65,160,80) );
    WarGlobals::pPartManager->addSystem( blood );
  }
  return true;
}

void SlimeAlien::kill()
{
}

void SlimeAlien::makeSound( GameObject::SoundID aSoundId ) const
{
}

ObjectID::Type SlimeAlien::objectType() const
{
  return ObjectID::TYPE_SLIMEALIEN;
}

void SlimeAlien::attack()
{
  if ( this->reloading() == true ) {
    return;
  }
  
  
  ObjectID::Type oid = ObjectID::TYPE_SLIMEALIEN;
  float damage = Settings::floatObjProp(oid, "damage:");
  int reload = Settings::intObjProp(oid, "reload:");
  
  GameObject* player = WarGlobals::pObjTable->pPlayer;
  Bullet* tmpBullet = new Bullet( this, Bullet::ERifle, player->position() );
  tmpBullet->iDamage = damage;
  player->hitByBullet( tmpBullet );
  delete tmpBullet;
  
  this->setCounter( Alien::RELOAD_COUNTER_INDEX, reload );
}




///
/// New public methods
/// ============================================================================




///
/// Getter methods
/// ============================================================================




///
/// Protected and private methods
/// ============================================================================

void SlimeAlien::setCorrectAnimation( int aAnimId )
{
  if ( this->getAnimID() != aAnimId ) {
    const Animation& anim = GameAnims::findAnimation( AnimId::KSlimeAlien, aAnimId );
    this->setAnimation( anim, aAnimId );
  }
}


};  // end of namespace
